//import angular core
import { Injectable } from '@angular/core';
import { Http, Headers, Response } from '@angular/http';

//import third package
import { Observable }     from 'rxjs/Observable';
import 'rxjs/add/operator/map';
import 'rxjs/add/operator/toPromise';

//import my services
import { ApiService } from './api.service';

//import class
import { Hero } from '../domain/hero';

@Injectable()
export class HeroService {
  private api_url : string ;
  private headers : Headers ;

  constructor(private http: Http, private apiService: ApiService) {
      this.api_url = apiService.getUrl() + '/heroes';
      this.headers = apiService.getHeaders();
  }

  //查询所有Hero
  getHeroes(): Promise<Hero[]> {
    const url = `${this.api_url}`;
    return this.http.get(url, { headers: this.headers })
      .toPromise()
      .then(res => res.json() as Hero[])
      .catch(this.handleError);
  }

  //按id查询Hero
  getHeroById(id: number): Promise<Hero> {
    const url = `${this.api_url}/${id}`;
    return this.http.get(url, { headers: this.headers })
      .toPromise()
      .then(res => res.json() as Hero)
      .catch(this.handleError);
  }

  //按ownerId查询Hero
  getHeroesByOwnerId(ownerId: number): Promise<Hero[]> {
    const url = `${this.api_url}/?ownerId=${ownerId}`;
    return this.http.get(url, { headers: this.headers })
      .toPromise()
      .then(res => res.json() as Hero[])
      .catch(this.handleError);
  }

  //search hero
  searchHero(term: string): Observable<Hero[]> {
    return this.http
      .get(`${this.api_url}?name_like=${term}`)
      .map(response => response.json() as Hero[]);
  }

  //新建Hero
  createHero(hero: Hero): Promise<Hero> {
    const url = `${this.api_url}`;
    return this.http
      .post(url, JSON.stringify(hero), { headers: this.headers })
      .toPromise()
      .then(res => res.json() as Hero)
      .catch(this.handleError);
  }

  //按name与userId新建Hero
  createHeroByNameUserId(name: string, ownerId: number): Promise<Hero> {
    let hero = {
      name: name,
      ownerId: ownerId
    }
    const url = `${this.api_url}`;
    return this.http
      .post(url, JSON.stringify(hero), { headers: this.headers })
      .toPromise()
      .then(res => res.json() as Hero)
      .catch(this.handleError);
  }

  //修改Hero
  updateHero(hero: Hero): Promise<Hero> {
    const url = `${this.api_url}/${hero.id}`;
    return this.http
      .put(url, JSON.stringify(hero), { headers: this.headers })
      .toPromise()
      .then(res => res.json() as Hero)
      .catch(this.handleError);
  }

  //删除某个Hero
  deleteHero(hero: Hero): Promise<void> {
    const url = `${this.api_url}/${hero.id}`;
    return this.http.delete(url, { headers: this.headers })
      .toPromise()
      .then(() => null)
      .catch(this.handleError);
  }

  //按id删除某个Hero
  deleteHeroById(id: number): Promise<void> {
    const url = `${this.api_url}/${id}`;
    return this.http.delete(url, { headers: this.headers })
      .toPromise()
      .then(() => null)
      .catch(this.handleError);
  }

  private handleError(error: any): Promise<any> {
    console.error('An error occurred', error);
    return Promise.reject(error.message || error);
  }
}
